<?php

declare(strict_types=1);

namespace NowPaymentsPlugin\Service;

use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionStateHandler;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Shopware\Core\Checkout\Payment\Cart\AsyncPaymentTransactionStruct;
use Shopware\Core\Checkout\Payment\Cart\PaymentHandler\AsynchronousPaymentHandlerInterface;
use Shopware\Core\Checkout\Payment\Exception\AsyncPaymentProcessException;
use Shopware\Core\Checkout\Payment\Exception\CustomerCanceledAsyncPaymentException;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\RouterInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use NowPaymentsPlugin\Util\DebugLog;


class NowPayment implements AsynchronousPaymentHandlerInterface
{
    /**
     * @var OrderTransactionStateHandler
     */
    private $transactionStateHandler;

    /**
     * @var SystemConfigService
     */
    private $systemConfigService;

    /**
     * @var EntityRepositoryInterface
     */
    private $currencyRepository;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var DebugLog
     */
    private $debugLog;

    public function __construct(OrderTransactionStateHandler $transactionStateHandler, EntityRepositoryInterface $currencyRepository, RouterInterface $router, SystemConfigService $systemConfigService, DebugLog $debugLog)
    {
        $this->transactionStateHandler = $transactionStateHandler;
        $this->systemConfigService = $systemConfigService;
        $this->currencyRepository = $currencyRepository;
        $this->router = $router;
        $this->debugLog = $debugLog;
    }


    /**
     * @param array $payment_data
     * @return string
     */
    public function createPaymentToken($payment_data)
    {
        unset($payment_data["successURL"]);
        unset($payment_data["cancelURL"]);
        unset($payment_data["products"]);
        return sha1(implode('|', $payment_data));
    }

    /**
     * @param array $orderEntity
     * @return string
     */
    public function getProductsArray($LineItems)
    {
        $NowPayLineItems = [];
        try {
            foreach ($LineItems as $item) {

                $Nowpayitem = [
                    "id" => $item->getIdentifier(),
                    "order_id" => $item->getOrderId(),
                    "name" => $item->getLabel(),
                    "product_id" => $item->getProductId(),
                    "variation_id" => $item->getReferencedId(),
                    "quantity" => $item->getQuantity(),
                    "tax_class" => "",
                    "subtotal" => round($item->getTotalPrice(), 6),
                    "subtotal_tax" => "0",
                    "total" => round($item->getTotalPrice(), 6),
                    "total_tax" => "0"
                ];
                array_push($NowPayLineItems, $Nowpayitem);
            }
        } catch (\Exception $e) {
            return $NowPayLineItems;
        }

        return $NowPayLineItems;
    }

    private function getPaymentData(AsyncPaymentTransactionStruct $transaction, SalesChannelContext $salesChannelContext)
    {

        $orderEntity = $transaction->getOrder();
        $ReturnUrl = $transaction->getReturnUrl();
        try {
            $currencyResult = $this->currencyRepository->search(new Criteria([$orderEntity->getCurrencyId()]), $salesChannelContext->getContext());
            /** @var CurrencyEntity $currencyEntity */
            $currencyEntity = array_pop($currencyResult->getElements());
            $currencyCode = $currencyEntity->getShortName();
        } catch (\Exception $e) {
            $currencyCode = 'EUR';
        }

        if (!empty($ReturnUrl)) {
            parse_str(parse_url($ReturnUrl, PHP_URL_QUERY), $returnQuery);

            $callBackUrl = $this->router->generate('frontend.checkout.nowpayment.callback', ['_sw_order' => $transaction->getOrderTransaction()->getId()], 0);
            $successURL = $this->router->generate('frontend.checkout.nowpayment.process', array_replace(['_sw_order' => $transaction->getOrderTransaction()->getId()], $returnQuery), 0);
            $cancelURL = $this->router->generate('frontend.checkout.nowpayment.process', array_replace(['_sw_order' => $transaction->getOrderTransaction()->getId()], $returnQuery), 0);
        } else {
            $callBackUrl = '';
            $successURL = '';
            $cancelURL = '';
        }

        $parameters = [
            'dataSource' => ("Shopware"),
            'paymentCurrency' => strtoupper($currencyCode),
            'successURL' => $successURL,
            'ipnURL' => $callBackUrl,
            'cancelURL' =>  $cancelURL,
            'orderID' => $salesChannelContext->getSalesChannel()->getName() . "_" . $orderEntity->getOrderNumber(),
            'customerName' => ((bool) $this->systemConfigService->get('NowPaymentsPlugin.config.transmitCustomerData')
                ? ($orderEntity->getOrderCustomer()->getFirstName()) . " " . $orderEntity->getOrderCustomer()->getLastName()
                : ""),
            'customerEmail' => ((bool) $this->systemConfigService->get('NowPaymentsPlugin.config.transmitCustomerData')
                ? $orderEntity->getOrderCustomer()->getEmail()
                : ""),
            'payment_id' => $transaction->getOrderTransaction()->getId(),
            'paymentAmount' => round($orderEntity->getAmountTotal(), 6),
            'products' => ((bool) $this->systemConfigService->get('NowPaymentsPlugin.config.transmitProductData')
                ? $this->getProductsArray($orderEntity->getLineItems())
                : "")
        ];
        return $parameters;
    }

    public function createPaymentUrl($parameters, $version)
    {
        $apiUrl = $this->systemConfigService->get('NowPaymentsPlugin.config.apiUrl');
        $apiKey = $this->systemConfigService->get('NowPaymentsPlugin.config.apiKey');

        if (empty($apiUrl)) {
            $this->logger->error('[NowPaymentsPlugin] ApiURL missing');
            return false;
        }
        if (empty($apiKey)) {
            $this->logger->error('[NowPaymentsPlugin] ApiKey missing');
            return false;
        }
        $parameters['token'] = $this->createPaymentToken($parameters);
        $parameters['apiKey'] = $apiKey;
        $parameters["plugin_version"] = $version;

        return $apiUrl . "?data=" . urlencode(json_encode($parameters));
    }

    public function validatePayment($paymentResponse)
    {
        //TODO:Coress refrence woth payment gateway 
        //curl --location --request GET 'https://api.nowpayments.io/v1/payment/5524759814' \
        //--header 'x-api-key: A7M40XV-CG1448Z-KVVED3G-NW3V0TK'
        return true;
    }

    public function validateApiKey()
    {
        if ($this->systemConfigService->get('NowPaymentsPlugin.config.apiKey')) {
            //TODO:verify Api with by post to url 
            //curl --location --request GET 'https://api.nowpayments.io/v1/currencies' \
            //--header 'x-api-key: A7M40XV-CG1448Z-KVVED3G-NW3V0TK'
            return true;
        }
        return false;
    }
    public function validateIPNSecret()
    {
        if ($this->systemConfigService->get('NowPaymentsPlugin.config.IPNSecret')) {
            //TODO:verify IPN with by post to url , aks clinet
            return true;
        }
        return false;
    }
    /**
     * @throws AsyncPaymentProcessException
     */
    public function pay(
        AsyncPaymentTransactionStruct $transaction,
        RequestDataBag $dataBag,
        SalesChannelContext $salesChannelContext
    ): RedirectResponse {

        $CitvApi = $this->validateApiKey();
        $CitvIPN = $this->validateIPNSecret();

        if (!$CitvApi || !$CitvIPN) {

            $errormsg = 'Please Contact Support there has been an Error. ';
            if (!$CitvApi) {
                $errormsg .= '<br>- nowpayment.io API Key is not Set.';
            }
            if (!$CitvIPN) {
                $errormsg .= '<br>- nowpayment.io IPN Secret is not Set.';
            }

            $route =  $this->router->generate('frontend.checkout.nowpayment.error', ['message' => $errormsg], 0);
            return new RedirectResponse($route, 302);

            //return $this->forwardToRoute( 'frontend.checkout.nowpayment.error',  ['message' => $errormsg]);

            //$route = $salesChannelContext->getSalesChannel()->getDomains()->first()->getUrl() . '/checkout/nowpayment/error?message=' . urlencode($errormsg);
            //return new RedirectResponse($route, 302);
        }

        try {
            $paymentData = $this->getPaymentData($transaction, $salesChannelContext);
            $this->debugLog->send('paymentData', $paymentData);
            $redirectUrl = $this->createPaymentUrl($paymentData, $salesChannelContext->getContext()->getVersionId());
            $this->debugLog->send('redirectUrl', $redirectUrl);

            if (!$redirectUrl) {
                $this->debugLog->send('Error', 'An error occurred during the communication with external payment gateway');
                throw new AsyncPaymentProcessException(
                    $transaction->getOrderTransaction()->getId(),
                    'An error occurred during the communication with external payment gateway' . PHP_EOL
                );
            }
        } catch (\Exception $e) {
            $this->debugLog->send('Error', 'An error occurred during the communication with external payment gateway' . PHP_EOL . $e->getMessage());
            throw new AsyncPaymentProcessException(
                $transaction->getOrderTransaction()->getId(),
                'An error occurred during the communication with external payment gateway' . PHP_EOL . $e->getMessage()
            );
        }

        // Redirect to external gateway
        $this->transactionStateHandler->process($transaction->getOrderTransaction()->getId(), $salesChannelContext->getContext());
        $this->debugLog->send('Redirect to external gateway', date('m/d/Y h:i:s a', time()));
        return new RedirectResponse($redirectUrl);
    }

    function isJson($string)
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    /**
     * @throws CustomerCanceledAsyncPaymentException
     */
    public function finalize(
        AsyncPaymentTransactionStruct $transaction,
        Request $request,
        SalesChannelContext $salesChannelContext
    ): void {

        // try{

        $requestqueryaRRAY = array_replace($request->request->all(), $request->query->all());
        $requestContent = $request->getContent();
        $requestContent = $this->isJson($requestContent) ? json_decode($request->getContent(), true) : [];
        $response = array_replace($requestqueryaRRAY, $requestContent);

        //$response = json_decode($request->getContent(), true);
        $paymentState = $response['payment_status'];
        $context = $salesChannelContext->getContext();

        if ($paymentState === 'finished' || $paymentState === 'sending') {
            if ($this->validatePayment($response['payment_status'])) {
                $this->transactionStateHandler->paid($transaction->getOrderTransaction()->getId(), $context);
            }
        } else if ($paymentState === 'partially_paid') {
            $this->transactionStateHandler->payPartially($transaction->getOrderTransaction()->getId(), $context);
        } else if ($paymentState === 'failed' || $paymentState === 'refunded' || $paymentState === 'expired') {
            $this->transactionStateHandler->fail($transaction->getOrderTransaction()->getId(), $context);
        } else if ($paymentState === 'confirming' || $paymentState === 'confirmed' || $paymentState === 'sending') {
            // $this->transactionStateHandler->process($transaction->getOrderTransaction()->getId(), $context);
        }
        //     } catch (\Exception $e) {
        //     $this->debugLog->send('Error', 'An error occurred during reading the gateway responce' . PHP_EOL . $e->getMessage());
        //     throw new AsyncPaymentProcessException(
        //         $transaction->getOrderTransaction()->getId(),
        //         'An error occurred during reading the gateway responce' . PHP_EOL . $e->getMessage()
        //     );
        // }
    }

    public function isValidToken($response_token, $token)
    {
        return hash_equals($token, $response_token);
    }
}
